#version 330
#extension GL_EXT_gpu_shader4 : enable
//TrianglesMod01.fsh by  amhall
//https://www.shadertoy.com/view/3sXczj
// Licence CC0
// Adapted, trivialy, for use in VGHD player
/////////////////////////////////////////////
uniform float u_Elapsed;    // The elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels

#define iTime u_Elapsed*0.314159  //*0.1666
#define iResolution u_WindowSize

//#define mouse AUTO_MOUSE
//#define MOUSE_SPEED vec2(vec2(0.5,0.577777) * 0.25)
//#define MOUSE_POS   vec2((1.0+cos(iTime*MOUSE_SPEED))*u_WindowSize/2.0)
//#define MOUSE_PRESS vec2(0.0,0.0)
//#define AUTO_MOUSE  vec4( MOUSE_POS, MOUSE_PRESS )
//#define RIGID_SCROLL
// alternatively use static mouse definition
#define iMouse vec4(0.0,0.0, 0.0,0.0)
//#define iMouse vec4(512,256,180,120)
uniform sampler2D iChannel0;
uniform sampler2D iChannel1;
uniform sampler2D iChannel2;
uniform sampler2D iChannel3;
vec4 texture2D_Fract(sampler2D sampler,vec2 P) {return texture2D(sampler,fract(P));}
vec4 texture2D_Fract(sampler2D sampler,vec2 P, float Bias) {return texture2D(sampler,fract(P),Bias);}
#define texture2D texture2D_Fract

// Created by Anthony Hall, March - July 2020
// The background can be replaced by a sampled texture which is pretty neat - 
// see getBackgroundColor

// Reducing supersampling is the best way to improve performance
// Disabling the rendering of the secondary triangles (line 57)
// also speeds things up quite a bit

const float pi = radians(180.0);
const float twoPi = 2.0 * pi;
const float sqrt3 = sqrt(3.0);

const int SUPERSAMPLE = 2; // Decrease to speed things up, increase to reduce aliasing
const float increment = 1.0 / float(SUPERSAMPLE);
const float offset = increment / 2.0 - 0.5;

// Primary triangles are the big ones that recurse on the whole scene
// Secondary triangles are the accessory ones that recurse just on themselves and glow
// This distinction and the large amount of globals they need lead to some long names

const float borderWidth = 0.0022;

// Parameters for primary recursion
const int maxIterations = 12; // This does not include iterations that are skipped when it zooms in
float scalePerIteration; // set in mainImage
float anglePerIteration; // set in mainImage
float zoom; // set in mainImage

// This is how long it takes to complete a cycle
// for the zoom amount per iteration
// The overall zoom takes twice as long as this
float secondsPerIteration = 30.0;

// Primary triangle geometry
const float primaryLength = 0.72; // side length
const float primaryRadius = 0.7; // radius of triangle center from origin

// The angle offset that the first triangle will be
// Other triangles are 120 degrees apart
const float primaryAngleOffset = 30.0;

const vec2[] triangleCenters = vec2[](
	vec2(0.0, 0.0),
    
	primaryRadius * vec2(
        cos(radians(primaryAngleOffset)),
        sin(radians(primaryAngleOffset))),
    
	primaryRadius * vec2(
        cos(radians(primaryAngleOffset + 120.0)),
        sin(radians(primaryAngleOffset + 120.0))),
    
	primaryRadius * vec2(
        cos(radians(primaryAngleOffset + 240.0)),
		sin(radians(primaryAngleOffset + 240.0))));

// Parameters for secondary recursion
const bool renderSecondaryTriangles = true; // TODO Set to false to only render primary triangles - this greatly speeds up rendering
const int secondaryIterations = 3; // Iterations of the secondary triangle
const float secondaryShrinkage = 1.0 / float(secondaryIterations); // The linear shrinkage per iteration

const float secondaryLength = 0.25;
const float secondaryRadius = 0.82;
const float secondaryMirrorOffset = secondaryLength * 0.8;

// Angle offset that the first secondary triangle will be
const float secondaryAngleOffset = 90.0;

vec2[] secondaryTriangleCenters = vec2[](
    secondaryRadius * vec2(
        cos(radians(secondaryAngleOffset)),
		sin(radians(secondaryAngleOffset))),

    secondaryRadius * vec2(
        cos(radians(secondaryAngleOffset + 120.0)),
    	sin(radians(secondaryAngleOffset + 120.0))),

	secondaryRadius * vec2(
        cos(radians(secondaryAngleOffset + 240.0)),
		sin(radians(secondaryAngleOffset + 240.0))));


// Oscillates between min and max based on some function of time
// Starts at min when time is 0
float oscillate(float min, float max, float time) {
	float halfInterval = (max - min) / 2.0;
	float middle = min + halfInterval;
	return middle - halfInterval * cos(time);
}

// Returns 2D rotation matrix of angle in degrees
mat2 getRotationMatrix(float degrees) {
    float rads = radians(degrees);
    vec2 cs = vec2(cos(rads), sin(rads));
	return mat2(cs.x, -cs.y,
                cs.y, cs.x);
}

// Thanks for all the distance functions iq
// Modified to support rotated triangles
float sdEquilateralTriangle(vec2 point, vec2 center, float angle, float sideLength)
{
    float radius = sideLength/2.0;
    point = point - center;
    if (angle != 0.0) {
     	point = getRotationMatrix(angle) * point;   
    }
    const float k = sqrt3;

    point.x = abs(point.x) - radius;
    point.y = point.y + radius/k;
    
    if (point.x + k*point.y > 0.0)
        point = vec2(point.x - k*point.y, -k*point.x - point.y) / 2.0;
    
    point.x -= clamp( point.x, -2.0*radius, 0.0 );
    return -length(point)*sign(point.y);
}

// Get the background color as a radial pattern multiplied by a base color
vec3 getBackgroundColor(vec2 point, int level)
{    
    // TODO - Texture Mode
    // Sample a texture instead of the regular background
    // This is way cooler using a picture of space. To use that,
    // run the following line of code in your browser's javascript console,
    // or replace the image URL with your own:
    
    // gShaderToy.SetTexture(0, {mSrc:'https://images.pexels.com/photos/110854/pexels-photo-110854.jpeg?auto=compress&cs=tinysrgb&dpr=1', mType:'texture', mID:1, mSampler:{ filter: 'linear', wrap: 'repeat', vflip:'true', srgb:'false', internal:'byte' }});
    
    // Append */ to the following line to use texture mode:
    /* 
    float aspect = iChannelResolution[0].x / iChannelResolution[0].y;
    
    // Add some simple wavy distortion
    vec2 distort = sin(point * 4.8);
    point += 0.075 * distort * distort;
    
    // This basically makes it so that the texture scale cycles on [0.33, 0.53] based on the zoom per iteration
    // and moves the texture a little bit
    vec2 uv = (0.33 + 0.2*(scalePerIteration - 1.55)/3.55) * vec2(point.x/aspect, point.y) - 0.5 + iTime/90.0;
    
    // The following lines reflect adjacent textures
    uv = mod(uv, 2.0);
    uv = 1.0 - abs(uv - 1.0);
    
    return texture(iChannel0, uv).xyz; /**/
    
    // Original - Calculate base color as blue becoming bright purple/orange as recursion happens
    // vec3 baseColor = vec3(float(level) * 0.35, float(level) * 0.15, 0.9);

    // Oscillation-based version
    vec3 baseColor = vec3(
        oscillate(0.0, 0.8, float(level) * 1.1) + float(level) / float(maxIterations) / 3.0,
        oscillate(0.0, 0.5, float(level) * 0.8) + float(level) / float(maxIterations) / 3.0,
        oscillate(0.9, 0.0, float(level) * 0.5) + float(level) / float(maxIterations) / 1.5);
    
	float angle = atan(point.y, point.x);
    float distance = distance(point, vec2(0.0));
    float angleIntensity = (sin(9.0*angle) / 4.0) + 0.75;
    float distanceIntensity = (cos(10.0 * distance) / 4.0) + 0.75;
    return baseColor * distanceIntensity * angleIntensity;
}

// Get rainbow color of screen the same way a default shadertoy does
vec3 getRainbowColor(vec2 point) {
    return 0.5 + 0.5*cos(iTime+point.xyx+vec3(0,2,4));
}

// Gets the intensity of a primary triangle glow, clamped from 0 to 1
float getPrimaryGlowIntensity(float distanceFromTriangle, float distanceFromCenter)
{
    // First, set center distance to start at the edge
    distanceFromCenter -= primaryLength / (2.0 * sqrt3);
    // Get a shape between triangle and circle distance, then calculate some function that falls off
    float interpolatedDistance = mix(distanceFromTriangle, distanceFromCenter, 0.2);
    float falloff = 1.0 / (interpolatedDistance * 50.0 + 1.0) - 0.08;
    return  clamp(falloff, 0.0, 1.0);
}

// Gets the intensity of a secondary triangle glow, clamped from 0 to 1
float getSecondaryGlowIntensity(float distanceFromTriangle, float distanceFromCenter, int recursionLevel)
{
    // First, set center distance to start at the edge
    distanceFromCenter -= secondaryLength / (2.0 * sqrt3);
    
    // Scale distances to make glow intensity oscillate
    float scale = oscillate(0.5, 1.5, 0.5*pi*(iTime + 2.0*float(recursionLevel)/float(secondaryIterations - 1)));
    
    distanceFromTriangle /= scale;
    distanceFromCenter /= scale;
    
    // Get a shape between triangle and circle distance, then calculate some function that falls off
    float interpolatedDistance = mix(distanceFromTriangle, distanceFromCenter, 0.2);
    float falloff = 1.0 / (interpolatedDistance * 150.0 + 1.0) - 0.1;
    return  clamp(falloff, 0.0, 1.0);
}

// Calculates the new coordinate if we are inside a main triangle
vec2 recurseOnMainTriangle(vec2 point, mat2 rotationMatrix, int triangleIndex)
{
    return scalePerIteration * rotationMatrix * (point - triangleCenters[triangleIndex]);
}

// Calculates the new coordinate if we are inside a secondary triangle
vec2 recurseOnSecondaryTriangle(vec2 originalPoint, vec2 center, int recursionLevel)
{
    // Scale linearly based on level
    float scale = (1.0 - float(recursionLevel + 1) * secondaryShrinkage);
    return (originalPoint - center) / scale + center;
}

// For skipping iterations when zoomed in
const float inscribeRadius = primaryLength / 2.0 / sqrt3 - borderWidth;

// Main algorithm - computes color at point, with rotation matrix passed to save computations
vec3 getColor(vec2 coord, mat2 rotationMatrix)
{ 	
    // Point to recurse on
    vec2 point = coord;
    
    // If the point is close to the center, advance the start level to save computation
    float exponent = 0.5 * log(inscribeRadius * inscribeRadius / dot(point, point)) / log(scalePerIteration);
    float start = max(ceil(exponent), 0.0);
    
    point = getRotationMatrix(anglePerIteration * start) * point * pow(scalePerIteration, start);
    
    // Recurse a few levels deep, keep track of level for color calculation
    vec3 result;
    for (int level = int(start); level < maxIterations + int(start); level++)
    {
        // Calculate background with pattern and some rainbow action
        vec3 background = getBackgroundColor(point, level);
        vec3 rainbow = getRainbowColor(point);
        
        // Reset result at beginning of recursion
        result = mix(background, rainbow, 0.25);
        
        int insideTriangleIndex = -1; // Index of the triangle we are inside - -1 if outside all triangles
        
        // Iterate over each main triangle
        for (int i = 0; i < triangleCenters.length(); i++) {
            vec2 center = triangleCenters[i];
            
            float distanceFromCenter = distance(point, center);
            // Optimize - stop calculations for this triangle if we are far from it
            if(distanceFromCenter > 0.8) 
                continue;
            
            float distanceFromTriangle = sdEquilateralTriangle(point, center, 0.0, primaryLength);
            
            // Break to recurse on this triangle if inside 
            if(distanceFromTriangle < 0.0)
            {
                // Add border to triangle
                if(distanceFromTriangle > -borderWidth)
                    return vec3(1.0);
                
                insideTriangleIndex = i;
                break;
            }
            
            // Add glow to result color
            float glowIntensity = getPrimaryGlowIntensity(distanceFromTriangle, distanceFromCenter);
            result += vec3(glowIntensity);
        }
        
        // If inside a triangle, set up and perform recursion
        if (insideTriangleIndex >= 0) {
        	point = recurseOnMainTriangle(point, rotationMatrix, insideTriangleIndex);
            continue;
        }
        
        // Otherwise, render secondary triangles, then break recursion
        if (renderSecondaryTriangles) {
            for (int i = 0; i < secondaryTriangleCenters.length(); i++)
            {
                vec2 center = secondaryTriangleCenters[i];

                // Store original point, because recursion will calculate from the original point
                // rather than the previous iteration's point
                vec2 originalPoint = point;

                // Render mirror copies
                for(int mirror = 0; mirror < 2; mirror++)
                {
                    // Reset point to originalPoint for each mirror copy
                    point = originalPoint;

                    for (int secondaryLevel = 0; secondaryLevel < secondaryIterations; secondaryLevel++)
                    {
                        // Optimize - abort rendering if we are too far away
                        float distanceFromCenter = distance(point, center);
                        if(distanceFromCenter > 0.3)
                            break;

                        float distanceFromTriangle = sdEquilateralTriangle(point, center, 60.0 * float(mirror), secondaryLength); // Add 60 degrees on mirror

                        // We are inside triangle
                        if(distanceFromTriangle < 0.0)
                        {
                            // Add border to triangle
                            if(distanceFromTriangle > -borderWidth)
                                return vec3(1.0);

                            // Reset result
                            result = 0.5 * getRainbowColor(point);

                            // Set up recursion
                            point = recurseOnSecondaryTriangle(originalPoint, center, secondaryLevel);
                            continue;
                        }

                        // Add glow and break recursion if outside triangle
                        float glowIntensity = getSecondaryGlowIntensity(distanceFromTriangle, distanceFromCenter, secondaryLevel);

                        result += vec3(glowIntensity);
                        break;
                    }
                    // Subtract offset of second triangle
                    vec2 offset = secondaryMirrorOffset * normalize(center);
                    center -= offset;
                }
            }
        }
        break; // Break from primary triangle recursion (because we are outside a primary triangle)
    }
    return result;
}
void main (void)
//void mainImage( out vec4 fragColor, in vec2 fragCoord )
{
    // Smaller screen dimension will have coordinates on [-1, 1]
    float minDimension = min(iResolution.x, iResolution.y);
    
    // Set up angle, scale, and zoom
    scalePerIteration = oscillate(1.55, 5.0, iTime * twoPi / secondsPerIteration);
    zoom = exp(oscillate(0.0, 10.0, iTime * pi / secondsPerIteration));
    anglePerIteration = iTime * 20.0 / secondsPerIteration;
    
    // Angle/Zoom defaults overwritten by mouse x/y respectively if clicked
    if (iMouse.z > 0.0)
    {
        vec2 mouse = min(iMouse.xy, iResolution.xy); // so changing resolution doesn't waz it
        anglePerIteration = -60.0 * (2.0 * mouse.x - iResolution.x) / iResolution.x;
        scalePerIteration = mix(1.55, 5.0, pow(exp2(mouse.y / iResolution.y) - 1.0, 1.5));
    }
    mat2 rotationMatrix = getRotationMatrix(anglePerIteration);
    
    // Supersample by accumulating color of all samples
    vec3 color = vec3(0.0);
    
    for(int j = 0; j < SUPERSAMPLE; j++) {
        for(int i = 0; i < SUPERSAMPLE; i++)
        {
            vec2 screenCoord = gl_FragCoord.xy + offset + increment * vec2(float(i), float(j));
            vec2 coord = 2.0 * (screenCoord - iResolution.xy/2.0) / minDimension;

            // Zoom and rotate
            coord = rotationMatrix * coord / zoom;

            color += getColor(coord, rotationMatrix);
        }
    }
    color /= float(SUPERSAMPLE * SUPERSAMPLE);

    // Output to screen
    gl_FragColor = vec4(color, 1.0);
}